<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Traits\CsvImportTrait;
use App\Http\Requests\MassDestroySchReportRequest;
use App\Http\Requests\StoreSchReportRequest;
use App\Http\Requests\UpdateSchReportRequest;
use App\Models\SchClass;
use App\Models\SchReport;
use App\Models\Schstudent;
use App\Models\SchSubject;
use App\Models\SchTeacher;
use Gate;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class SchReportController extends Controller
{
    use CsvImportTrait;

    public function index()
    {
        abort_if(Gate::denies('sch_report_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $schReports = SchReport::with(['schclass', 'schstudent', 'schteacher', 'schsubject'])->get();

        $sch_classes = SchClass::get();

        $schstudents = Schstudent::get();

        $sch_teachers = SchTeacher::get();

        $sch_subjects = SchSubject::get();

        return view('admin.schReports.index', compact('schReports', 'sch_classes', 'sch_subjects', 'sch_teachers', 'schstudents'));
    }

    public function create(Request $request)
    {
        abort_if(Gate::denies('sch_report_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $classId = $request->input('class_id'); // Отримуємо class_id з URL

        $schclasses = SchClass::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schstudents = Schstudent::pluck('name_ua', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schteachers = SchTeacher::pluck('name_ua', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schsubjects = SchSubject::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        return view('admin.schReports.create', compact('schclasses', 'schstudents', 'schsubjects', 'schteachers', 'classId'));
    }

    public function store(StoreSchReportRequest $request)
    {
        $schReport = SchReport::create($request->all());

        $data = $request->validate([
            'class_id' => 'required|exists:sch_classes,id',
            'report' => 'required|string',
        ]);

        SchReport::create($data);

        return redirect()->route('admin.sch-reports.index')->with('success', 'Репорт створено успішно!');
    }

    public function edit(SchReport $schReport)
    {
        abort_if(Gate::denies('sch_report_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $schclasses = SchClass::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schstudents = Schstudent::pluck('name_ua', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schteachers = SchTeacher::pluck('name_ua', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schsubjects = SchSubject::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        $schReport->load('schclass', 'schstudent', 'schteacher', 'schsubject');

        return view('admin.schReports.edit', compact('schReport', 'schclasses', 'schstudents', 'schsubjects', 'schteachers'));
    }

    public function update(UpdateSchReportRequest $request, SchReport $schReport)
    {
        $schReport->update($request->all());

        return redirect()->route('admin.sch-reports.index');
    }

    public function show(SchReport $schReport)
    {
        abort_if(Gate::denies('sch_report_show'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $schReport->load('schclass', 'schstudent', 'schteacher', 'schsubject');

        return view('admin.schReports.show', compact('schReport'));
    }

    public function destroy(SchReport $schReport)
    {
        abort_if(Gate::denies('sch_report_delete'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $schReport->delete();

        return back();
    }

    public function massDestroy(MassDestroySchReportRequest $request)
    {
        $schReports = SchReport::find(request('ids'));

        foreach ($schReports as $schReport) {
            $schReport->delete();
        }

        return response(null, Response::HTTP_NO_CONTENT);
    }

    public function getClassData(Request $request)
    {
        $classId = $request->input('class_id');

        // Вибираємо та сортуємо учнів, вчителів і предмети по алфавіту
        $students = \App\Models\SchStudent::where('schclass_id', $classId)
            ->orderBy('name')
            ->pluck('name', 'id');

        $teachers = \App\Models\SchTeacher::where('schclass_id', $classId)
            ->orderBy('name')
            ->pluck('name', 'id');

        $subjects = \App\Models\SchSubject::where('schclass_id', $classId)
            ->orderBy('name')
            ->pluck('name', 'id');

        return response()->json([
            'students' => $students,
            'teachers' => $teachers,
            'subjects' => $subjects,
        ]);
    }

    // filepath: app/Http/Controllers/Admin/SchReportController.php
    public function classList()
    {
        $schclasses = \App\Models\SchClass::orderBy('name')->get();
        return view('admin.schReports.class_list', compact('schclasses'));
    }
}
